package main

import (
	"math/rand"
	"runtime"
	"time"

	"github.com/veandco/go-sdl2/img"
	"github.com/veandco/go-sdl2/mix"
	"github.com/veandco/go-sdl2/sdl"
	"github.com/veandco/go-sdl2/ttf"
)

var running = true
var screenWidth int32
var screenHeight int32

func init() {
	runtime.LockOSThread()
}

func main() {
	rand.Seed(time.Now().UTC().UnixNano())
	sdl.Init(sdl.INIT_EVERYTHING)
	img.Init(img.INIT_PNG)
	ttf.Init()
	mix.Init(mix.INIT_OGG)

	var windowFlags uint32 = sdl.WINDOW_FULLSCREEN_DESKTOP

	window, err := sdl.CreateWindow("snosscire - birdie 2023 demo",
		sdl.WINDOWPOS_UNDEFINED,
		sdl.WINDOWPOS_UNDEFINED,
		1280, 720,
		windowFlags)
	if err != nil {
		panic(err)
	}
	defer window.Destroy()

	renderer, err := sdl.CreateRenderer(window, -1, sdl.RENDERER_ACCELERATED)
	if err != nil {
		panic(err)
	}
	defer renderer.Destroy()

	screenWidth, screenHeight, err = renderer.GetOutputSize()
	if err != nil {
		panic(err)
	}

	err = mix.OpenAudio(
		mix.DEFAULT_FREQUENCY,
		mix.DEFAULT_FORMAT,
		mix.DEFAULT_CHANNELS,
		mix.DEFAULT_CHUNKSIZE)
	if err != nil {
		panic(err)
	}

	music := loadMusic()
	music.Play(-1)

	background1 := NewBackground(100, 1, 1, 0.2)
	background2 := NewBackground(10, 1, 1, 0.3)
	birdie1 := NewRotateAnimation(renderer, loadBirdie(), 0.2)
	birdie2 := NewRotateAnimation(renderer, loadBirdie(), -0.2)

	rymdskepp := NewRymdskepp(renderer)
	rymdskepp.x = screenWidth/2 - rymdskepp.Width()/2
	rymdskepp.y = screenHeight + rymdskepp.Height()

	var currentText *FaceInFadeOut
	var showRymdskeppAnim *ShowRymdskepp
	var asteroidField *AsteroidField
	var rymdskeppExplosion *ExplosionParticleEffect

	endDemo := func() {
		running = false
	}
	text10 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "vote.birdie.org", GetRandomColor()), 0.2, endDemo)
	}
	text9 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "...att rösta!", GetRandomColor()), 0.2, text10)
	}
	text8 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "Glöm inte...", GetRandomColor()), 0.2, text9)
	}
	text7 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "Nej vänta!", GetRandomColor()), 0.2, text8)
	}
	text6 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "Hej då!", GetRandomColor()), 0.2, text7)
	}

	afterExplosion := func() {
		rymdskeppExplosion = nil
		currentText = NewFadeInFadeOut(NewText(loadFont(), "Slut!", GetRandomColor()), 0.2, text6)
	}

	explodeRymdskepp := func() {
		showRymdskeppAnim = nil
		rymdskepp.show = false
		rymdskeppExplosion = NewExplosionParticleEffect(
			float32(rymdskepp.x+32),
			float32(rymdskepp.y+32),
			afterExplosion)
	}

	pilotRymdskepp := func() {
		asteroidField = NewAsteroidField(renderer, explodeRymdskepp)
		showRymdskeppAnim = nil
	}
	showRymdskeppAnim = NewShowRymdskepp(rymdskepp, 0.2, pilotRymdskepp)

	//showRymdskeppAnim.SetShow(true)
	showRymdskepp := func() {
		showRymdskeppAnim.SetShow(true)
	}

	text5 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "...som exploderade.", GetRandomColor()), 0.2, showRymdskepp)
	}
	text4 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "...om ett rymdskepp...", GetRandomColor()), 0.2, text5)
	}
	text3 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "...till mitt demo...", GetRandomColor()), 0.2, text4)
	}
	text2 := func() {
		currentText = NewFadeInFadeOut(NewText(loadFont(), "Välkommen...", GetRandomColor()), 0.2, text3)
	}
	currentText = NewFadeInFadeOut(NewText(loadFont(), "Hej!", GetRandomColor()), 0.2, text2)

	currentTime := sdl.GetTicks()
	lastTime := currentTime
	var deltaTime float32

	for running {
		currentTime = sdl.GetTicks()
		deltaTime = float32(currentTime - lastTime)
		lastTime = currentTime

		handleEvents()

		background1.Update(deltaTime)
		background2.Update(deltaTime)
		birdie1.Update(renderer, deltaTime)
		birdie2.Update(renderer, deltaTime)

		if asteroidField != nil {
			asteroidField.Update(deltaTime)
		}
		if showRymdskeppAnim != nil {
			showRymdskeppAnim.Update(deltaTime)
		}
		rymdskepp.Update(deltaTime)

		if rymdskeppExplosion != nil {
			rymdskeppExplosion.Update(deltaTime)
		}

		if currentText != nil {
			currentText.Update(deltaTime)
		}

		renderer.SetDrawColor(0, 0, 0, 255)
		renderer.Clear()

		background1.Draw(renderer)
		background2.Draw(renderer)
		birdie1.Draw(renderer, 0+10, 0+10)
		birdie2.Draw(renderer, screenWidth-birdie2.Width()-10, 0+10)

		if asteroidField != nil {
			asteroidField.Draw(renderer)
		}

		rymdskepp.Draw(renderer)

		if rymdskeppExplosion != nil {
			rymdskeppExplosion.Draw(renderer)
		}

		if currentText != nil {
			currentText.DrawCenter(renderer,
				screenWidth/2+int32(RandomNumber(-5, 5)),
				screenHeight/2+int32(RandomNumber(-5, 5)))
		}

		renderer.Present()
	}

	music.Free()
	mix.CloseAudio()

	mix.Quit()
	ttf.Quit()
	img.Quit()
	sdl.Quit()
}

func handleEvents() {
	for event := sdl.PollEvent(); event != nil; event = sdl.PollEvent() {
		switch t := event.(type) {
		case *sdl.QuitEvent:
			running = false
		case *sdl.KeyboardEvent:
			if t.Keysym.Sym == sdl.K_ESCAPE {
				running = false
			}
		}
	}
}

var musicAsset []byte
var musicRWops *sdl.RWops

func loadMusic() *mix.Music {
	var err error
	musicAsset, err = Asset("res/music.wav")
	if err != nil {
		panic(err)
	}
	musicRWops, err = sdl.RWFromMem(musicAsset)
	if err != nil {
		panic(err)
	}

	music, err := mix.LoadMUSTypeRW(musicRWops, mix.WAV, 0)
	if err != nil {
		panic(err)
	}

	return music
}

var birdieAsset []byte
var birdieRWops *sdl.RWops
var birdieSurface *sdl.Surface

func loadBirdie() *sdl.Surface {
	if birdieSurface == nil {
		var err error
		birdieAsset, err = Asset("res/birdie.png")
		if err != nil {
			panic(err)
		}
		birdieRWops, err = sdl.RWFromMem(birdieAsset)
		if err != nil {
			panic(err)
		}

		birdieSurface, err = img.LoadPNGRW(birdieRWops)
		if err != nil {
			panic(err)
		}
	}
	return birdieSurface
}

var fontAsset []byte
var fontRWops *sdl.RWops
var font *ttf.Font

func loadFont() *ttf.Font {
	if font == nil {
		var err error
		fontAsset, err = Asset("res/font.ttf")
		if err != nil {
			panic(err)
		}
		fontRWops, err = sdl.RWFromMem(fontAsset)
		if err != nil {
			panic(err)
		}

		font, err = ttf.OpenFontRW(fontRWops, 0, 96)
		if err != nil {
			panic(err)
		}
	}
	return font
}
